﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Timers;
using System.Data;
using TvDatabase;
using System.ComponentModel;
using SetupTv.Sections;
using SetupTv;
using System.Net.Sockets;
using System.Net;
using TvControl;
using TvLibrary.Interfaces;
using TvEngine.PowerScheduler.Interfaces;
using NetworkHandler;

namespace TvEngine
{
    public class HomeServerPlugin : ITvServerPlugin
    {
        // Fields
        private Timer _ActiveClientsCheckTimer = new Timer();
        private int _CheckInterval;
        private bool _ExtensiveLogging;
        private int _hbActiveClients;
        private DataTable _hbClientsTable = new DataTable("HBMClients");
        private string _PluginAuthor = "Diggen & krikkit";
        private string _PluginName = "HomeServer";
        private string _PluginVersion = "0.0.6.0";
        private Timer _RHBMTimer = new Timer();
        private int _SecondsToRemoveClients;
        private TvBusinessLayer _tvbLayer = new TvBusinessLayer();
        private BackgroundWorker _udpServerBackgroundWorker = new BackgroundWorker();

        // Methods
        private void _RHBMTimer_Elapsed(object sender, EventArgs e)
        {
            string msg = this._tvbLayer.GetSetting("HomeServer RHBM Hosts", "").Value;
            this.Logger(msg, true);
            if (!(msg != ""))
            {
                this.Logger("Reversed HeartBeat: No Hosts in List", true);
            }
            else
            {
                foreach (string str2 in msg.Split(";".ToCharArray()))
                {
                    if (NetworkHandler.NetworkHandler.isComputerAvaible(str2))
                    {
                        this.Logger("Reversed HeartBeat: Found at least one active Host", false);
                        GlobalServiceProvider.Instance.Get<IPowerScheduler>().UserActivityDetected(DateTime.Now);
                        return;
                    }
                }
                this.Logger("Reversed HearBeat: No Host seems to be active", false);
            }
        }

        private void ActiveClientsCheckTimer_Elapsed(object sender, EventArgs e)
        {
            this.Logger("HeartBeatServer: check for active Clients", true);
            for (int i = 0; i <= (this._hbClientsTable.Rows.Count - 1); i++)
            {
                DateTime now = DateTime.Now;
                DateTime time2 = Convert.ToDateTime(this._hbClientsTable.Rows[i]["DateTime"]);
                TimeSpan span = (TimeSpan) (now - time2);
                this.Logger("HeartBeatServer: " + this._hbClientsTable.Rows[i]["IP"].ToString() + " - " + this._hbClientsTable.Rows[i]["Message"].ToString() + " - timespan:" + span.Seconds.ToString() + "s  - now:" + now.ToString() + " - last active:" + time2.ToString(), true);
                if (Convert.ToInt32(span.TotalSeconds) >= Convert.ToInt32(this._SecondsToRemoveClients))
                {
                    this.Logger("HeartBeatServer: client died at " + this._hbClientsTable.Rows[i]["DateTime"].ToString() + ": " + this._hbClientsTable.Rows[i]["IP"].ToString() + " - " + this._hbClientsTable.Rows[i]["Message"].ToString(), false);
                    this._hbClientsTable.Rows[i].Delete();
                }
            }
            this._hbActiveClients = this._hbClientsTable.Rows.Count;
            this.Logger("HeartBeatServer: active Clients: " + this._hbActiveClients.ToString(), false);
            if (this._hbActiveClients != 0)
            {
                try
                {
                    if (GlobalServiceProvider.Instance.IsRegistered<IPowerScheduler>())
                    {
                        this.Logger("HeartBeatServer: tell PowerScheduler UserActivity detected, " + DateTime.Now.ToString(), false);
                        GlobalServiceProvider.Instance.Get<IPowerScheduler>().UserActivityDetected(DateTime.Now);
                    }
                    else
                    {
                        TvLibrary.Log.Log.Error("HeartBeatServer: can't tell PowerScheduler that there is UserActivity, no IPowerScheduler", new object[0]);
                    }
                }
                catch (Exception exception)
                {
                    TvLibrary.Log.Log.Error("HeartBeatServer:something bad happens with IPowerScheduler: " + exception.Message.ToString(), new object[0]);
                }
            }
        }

        private void Logger(string msg, bool onlyExtensiv)
        {
            Setting setting = this._tvbLayer.GetSetting("HomeServer ExtensiveLogging", "false");
            this._ExtensiveLogging = Convert.ToBoolean(setting.Value);
            if (!onlyExtensiv || this._ExtensiveLogging)
            {
                TvLibrary.Log.Log.Info(msg, new object[0]);
            }
        }

        public void Start(IController controller)
        {
            if (!Convert.ToBoolean(this._tvbLayer.GetSetting("pluginPower Scheduler", "false").Value))
            {
                TvLibrary.Log.Log.Error("HeartBeatServer: PowerSchedule Plugin is off, won't start HeartBeatServer", new object[0]);
            }
            else
            {
                Setting setting = this._tvbLayer.GetSetting("HomeServer HBM DeclareDeath", "45");
                this._SecondsToRemoveClients = Convert.ToInt32(setting.Value);
                this.Logger("HeartBeatServer: declare client dead after " + this._SecondsToRemoveClients.ToString() + " Seconds", true);
                setting = this._tvbLayer.GetSetting("HomeServer HBM CheckInterval", "60");
                this._CheckInterval = Convert.ToInt32(setting.Value);
                this.Logger("HeartBeatServer: Check if there are clients alive after " + this._CheckInterval.ToString() + " Seconds", true);
                this._hbClientsTable.Columns.Add("DateTime", typeof(DateTime));
                this._hbClientsTable.Columns.Add("IP", typeof(string));
                this._hbClientsTable.Columns.Add("Nr", typeof(int));
                this._hbClientsTable.Columns.Add("Message", typeof(string));
                this._udpServerBackgroundWorker.WorkerReportsProgress = true;
                this._udpServerBackgroundWorker.WorkerSupportsCancellation = true;
                this._udpServerBackgroundWorker.DoWork += new DoWorkEventHandler(this.udpServerBackgroundWorker_DoWork);
                this._udpServerBackgroundWorker.ProgressChanged += new ProgressChangedEventHandler(this.udpServerBackgroundWorker_ProgressChanged);
                this._ActiveClientsCheckTimer.Enabled = false;
                this._ActiveClientsCheckTimer.AutoReset = true;
                this._ActiveClientsCheckTimer.Interval = this._CheckInterval * 0x3e8;
                this._ActiveClientsCheckTimer.Elapsed += new ElapsedEventHandler(this.ActiveClientsCheckTimer_Elapsed);
                this._RHBMTimer.Enabled = false;
                this._RHBMTimer.AutoReset = true;
                this._RHBMTimer.Interval = 60000.0;
                this._RHBMTimer.Elapsed += new ElapsedEventHandler(this._RHBMTimer_Elapsed);
                this.Logger("HeartBeatServer: starting BackgroundWorker and Timer", false);
                this._udpServerBackgroundWorker.RunWorkerAsync();
                this._ActiveClientsCheckTimer.Start();
                this._RHBMTimer.Start();
            }
        }

        public void Stop()
        {
            this.Logger("HeartBeatServer: will stop BackgroundWorker after next HeartBeat", false);
            this._udpServerBackgroundWorker.CancelAsync();
            this._ActiveClientsCheckTimer.Stop();
            this._hbActiveClients = 0;
            this._hbClientsTable.Clear();
            this._RHBMTimer.Stop();
        }

        private void udpServerBackgroundWorker_DoWork(object sender, EventArgs e)
        {
            HeartbeatResults userState = new HeartbeatResults();
            userState.Number = 0;
            try
            {
                Socket socket = new Socket(AddressFamily.InterNetwork, SocketType.Dgram, ProtocolType.Udp);
                IPEndPoint localEP = new IPEndPoint(IPAddress.Any, 0xfaa);
                socket.Bind(localEP);
                this.Logger("HeartBeatServer: listen at Port 4010", true);
                while (!this._udpServerBackgroundWorker.CancellationPending)
                {
                    byte[] buffer = new byte[0x1000];
                    EndPoint remoteEP = localEP;
                    socket.ReceiveFrom(buffer, ref remoteEP);
                    string str = Encoding.ASCII.GetString(buffer).Replace("\0", " ").Trim();
                    if (userState.Number == 100)
                    {
                        userState.Number = 0;
                    }
                    userState.Number++;
                    userState.ReceiveTime = DateTime.Now;
                    IPEndPoint point3 = remoteEP as IPEndPoint;
                    userState.IP = point3.Address.ToString();
                    userState.Port = point3.Port;
                    userState.Message = str.ToString();
                    this._udpServerBackgroundWorker.ReportProgress(userState.Number, userState);
                }
                this.Logger("HeartBeatServer: BackgroundWorker stopped", true);
            }
            catch (Exception exception)
            {
                TvLibrary.Log.Log.Error("HeartBeatServer: BackgroundWorker died (" + exception.Message.ToString() + ") :(", new object[0]);
            }
        }

        private void udpServerBackgroundWorker_ProgressChanged(object sender, ProgressChangedEventArgs e)
        {
            HeartbeatResults userState = e.UserState as HeartbeatResults;
            int progressPercentage = e.ProgressPercentage;
            bool flag = false;
            for (int i = 0; i < this._hbClientsTable.Rows.Count; i++)
            {
                if (userState.IP.ToString() == this._hbClientsTable.Rows[i]["IP"].ToString())
                {
                    this._hbClientsTable.Rows[i]["DateTime"] = userState.ReceiveTime;
                    this._hbClientsTable.Rows[i]["IP"] = userState.IP;
                    this._hbClientsTable.Rows[i]["Nr"] = userState.Number;
                    this._hbClientsTable.Rows[i]["Message"] = userState.Message;
                    flag = true;
                    this.Logger(string.Concat(new object[] { "HeartBeatServer: update client at ", userState.ReceiveTime, ": ", userState.IP, " - ", userState.Message }), true);
                    break;
                }
            }
            if (!flag)
            {
                this.Logger(string.Concat(new object[] { "HeartBeatServer: new client at ", userState.ReceiveTime, ": ", userState.IP, " - ", userState.Message }), false);
                this._hbClientsTable.Rows.Add(new object[] { userState.ReceiveTime, userState.IP, userState.Number, userState.Message });
            }
        }

        // Properties
        public string Author
        {
            get
            {
                return this._PluginAuthor;
            }
        }

        public bool MasterOnly
        {
            get
            {
                return false;
            }
        }

        public string Name
        {
            get
            {
                return this._PluginName;
            }
        }

        public SectionSettings Setup
        {
            get
            {
                return new HomeServerUserControl();
            }
        }

        public string Version
        {
            get
            {
                return this._PluginVersion;
            }
        }

        // Nested Types
        public class HeartbeatResults
        {
            // Properties
            public string IP { get; set; }

            public string Message { get; set; }

            public int Number { get; set; }

            public int Port { get; set; }

            public DateTime ReceiveTime { get; set; }
        }
    }

 

}

